﻿namespace Microsoft.Samples.PlanMyNight.Data.Caching
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using Microsoft.Samples.PlanMyNight.Entities;

    public class CachedReferenceRepository : IReferenceRepository
    {
        private const string ReferenceDataCacheContainer = "Reference";

        private readonly TimeSpan ReferenceDataExpiration = TimeSpan.FromDays(1);

        private readonly ICachingProvider cacheProvider;

        private readonly IReferenceRepository repository;

        public CachedReferenceRepository(ICachingProvider cacheProvider) :
            this(cacheProvider, new ReferenceRepository())
        {
        }

        public CachedReferenceRepository(ICachingProvider cacheProvider, IReferenceRepository repository)
        {
            this.cacheProvider = cacheProvider;
            this.repository = repository;
        }

        public IEnumerable<State> RetrieveStates()
        {
            var cacheKey = "states";

            IEnumerable<State> data = this.cacheProvider.Get(ReferenceDataCacheContainer, cacheKey) as IEnumerable<State>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.RetrieveStates();
            this.cacheProvider.Add(ReferenceDataCacheContainer, cacheKey, data, this.ReferenceDataExpiration);
            Trace.WriteLine("ADD-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public IEnumerable<string> RetrieveCities(string stateCode)
        {
            var cacheKey = "cities-" + stateCode.ToLowerInvariant();

            IEnumerable<string> data = this.cacheProvider.Get(ReferenceDataCacheContainer, cacheKey) as IEnumerable<string>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            var normalizedStateCode = stateCode.ToUpperInvariant();
            data = this.repository.RetrieveCities(normalizedStateCode);
            this.cacheProvider.Add(ReferenceDataCacheContainer, cacheKey, data, this.ReferenceDataExpiration);
            Trace.WriteLine("ADD-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public IEnumerable<string> RetrieveZipCodes(string stateCode)
        {
            var cacheKey = "zipcodes-" + stateCode.ToLowerInvariant();

            IEnumerable<string> data = this.cacheProvider.Get(ReferenceDataCacheContainer, cacheKey) as IEnumerable<string>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            var normalizedStateCode = stateCode.ToUpperInvariant();
            data = this.repository.RetrieveZipCodes(normalizedStateCode);
            this.cacheProvider.Add(ReferenceDataCacheContainer, cacheKey, data, this.ReferenceDataExpiration);
            Trace.WriteLine("ADD-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
            return data;
        }
    }
}
